<?PHP

#
#   Axis--UserFactory.php
#   An Meta-Object for Handling User Information
#
#   Copyright 2003 Axis Data
#   This code is free software that can be used or redistributed under the
#   terms of Version 2 of the GNU General Public License, as published by the
#   Free Software Foundation (http://www.fsf.org).
#
#   Author:  Edward Almasy (almasy@axisdata.com)
#
#   Part of the AxisPHP library v1.2.4
#   For more information see http://www.axisdata.com/AxisPHP/
#

require_once("Axis--User.php");


class UserFactory {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function __construct(&$SessionOrDb)
    {
        # if a session was passed in
        #if (is_object($SessionOrDb) && method_exists($SessionOrDb, "Session"))
		if (is_object($SessionOrDb) && method_exists($SessionOrDb, "DeleteExpiredSessions"))
        {
            # swipe database handle from session
            $this->DB =& $SessionOrDb->DB;

            # save session
            $this->Session =& $SessionOrDb;
        }
        # else if database handle was passed in
        #elseif (is_object($SessionOrDb) && method_exists($SessionOrDb, "Database"))
		elseif (is_object($SessionOrDb) && method_exists($SessionOrDb, "TablesAccessed"))
        {
            # save database handle
            $this->DB =& $SessionOrDb;

            # create session
            $this->Session = new Session($this->DB);
        }
        else
        {
            # error out
            $this->Result = U_ERROR;
            exit(1);
        }
    }

    # create new user (returns User object or array of error codes)
    function CreateNewUser(
            $UserName, $Password, $PasswordAgain, $EMail, $EMailAgain)
    {
        # check incoming values
        $ErrorCodes = $this->TestNewUserValues(
            $UserName, $Password, $PasswordAgain, $EMail, $EMailAgain);

        # if error found in incoming values return error codes to caller
        if (count($ErrorCodes)) {  return $ErrorCodes;  }

        # add user to database
        $UserName = User::NormalizeUserName($UserName);
        $this->DB->Query("INSERT INTO APUsers"
                ." (UserName, CreationDate)"
                ." VALUES ('".addslashes($UserName)."', NOW())");

        # create new user object
        $UserId = $this->DB->LastInsertId("APUsers");
        $User = new User($this->DB, (int)$UserId);

        # if new user object creation failed return error code to caller
        if ($User->Status() != U_OKAY) {  return array($User->Status());  }

        # set password and e-mail address
        $User->SetPassword($Password);
        $User->Set("EMail", $EMail);

        # return new user object to caller
        return $User;
    }

    # test new user creation values (returns array of error codes)
    function TestNewUserValues(
            $UserName, $Password, $PasswordAgain, $EMail, $EMailAgain)
    {
        $ErrorCodes = array();
        if (strlen(User::NormalizeUserName($UserName)) == 0)
            {  $ErrorCodes[] = U_EMPTYUSERNAME;  }
        elseif (!User::IsValidUserName($UserName))
            {  $ErrorCodes[] = U_ILLEGALUSERNAME;  }
        elseif ($this->UserNameExists($UserName))
            {  $ErrorCodes[] = U_DUPLICATEUSERNAME;  }

        if ($this->EMailAddressIsInUse($EMail))
            {  $ErrorCodes[] = U_DUPLICATEEMAIL;  }

        $FoundOtherPasswordError = FALSE;
        if (strlen(User::NormalizePassword($Password)) == 0)
        {  
            $ErrorCodes[] = U_EMPTYPASSWORD;  
            $FoundOtherPasswordError = TRUE;
        }
        elseif (!User::IsValidPassword($Password))
        {  
            $ErrorCodes[] = U_ILLEGALPASSWORD;  
            $FoundOtherPasswordError = TRUE;
        }

        if (strlen(User::NormalizePassword($PasswordAgain)) == 0)
        {  
            $ErrorCodes[] = U_EMPTYPASSWORDAGAIN;  
            $FoundOtherPasswordError = TRUE;
        }
        elseif (!User::IsValidPassword($PasswordAgain))
        {  
            $ErrorCodes[] = U_ILLEGALPASSWORDAGAIN;  
            $FoundOtherPasswordError = TRUE;
        }

        if ($FoundOtherPasswordError == FALSE)
        {
            if (User::NormalizePassword($Password) 
                    != User::NormalizePassword($PasswordAgain))
            {  
                $ErrorCodes[] = U_PASSWORDSDONTMATCH;  
            }
        }

        $FoundOtherEMailError = FALSE;
        if (strlen(User::NormalizeEMailAddress($EMail)) == 0)
        {  
            $ErrorCodes[] = U_EMPTYEMAIL;  
            $FoundOtherEMailError = TRUE;
        }
        elseif (!User::IsValidLookingEMailAddress($EMail))
        {  
            $ErrorCodes[] = U_ILLEGALEMAIL;  
            $FoundOtherEMailError = TRUE;
        }

        if (strlen(User::NormalizeEMailAddress($EMailAgain)) == 0)
        {  
            $ErrorCodes[] = U_EMPTYEMAILAGAIN;
            $FoundOtherEMailError = TRUE;
        }
        elseif (!User::IsValidLookingEMailAddress($EMailAgain))
        {  
            $ErrorCodes[] = U_ILLEGALEMAILAGAIN;  
            $FoundOtherEMailError = TRUE;
        }

        if ($FoundOtherEMailError == FALSE)
        {
            if (User::NormalizeEMailAddress($EMail) 
                    != User::NormalizeEMailAddress($EMailAgain))
            {  
                $ErrorCodes[] = U_EMAILSDONTMATCH;  
            }
        }

        return $ErrorCodes;
    }

    # return total number of users in system
    function GetUserCount()
    {
        return $this->DB->Query("SELECT COUNT(*) AS UserCount FROM APUsers", "UserCount");
    }

    # return total number of user that matched last GetMatchingUsers call
    # before the return size was limited
    function GetMatchingUserCount()
    {
        return $this->MatchingUserCount;
    }

    # return array of users currently logged in
    function GetLoggedInUsers()
    {
        # start with empty array (to prevent array errors)
        $ReturnValue = array();

        # load array of logged in user
        $UserIds = $this->Session->GetFromAllSessions("APUserId");

        # for each logged in user
        foreach ($UserIds as $UserId)
        {
            # load all data values for user
            $this->DB->Query("SELECT * FROM APUsers WHERE UserId = '".$UserId."'");
            $ReturnValue[$UserId] = $this->DB->FetchRow();
        }

        # return array of user data to caller
        return $ReturnValue;
    }

    # return array of user names who have the specified privileges
    # (array index is user IDs)
    function GetUsersWithPrivileges()
    {
        # start with query string that will return all users
        $QueryString = "SELECT DISTINCT APUsers.UserId, UserName FROM APUsers, APUserPrivileges";

        # for each specified privilege
        $Args = func_get_args();
        foreach ($Args as $Index => $Arg)
        {
            # add condition to query string
            $QueryString .= ($Index == 0) ? " WHERE (" : " OR";
            $QueryString .= " APUserPrivileges.Privilege = ".$Arg;
        }

        # close privilege condition in query string and add user ID condition
        $QueryString.= count($Args) ? ") AND APUsers.UserId = APUserPrivileges.UserId" : "";

        # perform query
        $this->DB->Query($QueryString);

        # copy query result into user info array
        $Users = $this->DB->FetchColumn("UserName", "UserId");

        # return array of users to caller
        return $Users;
    }

    # return array of user objects who have values matching search string
    # (array indexes are user IDs)
    function FindUsers($SearchString, $FieldName = "UserName", 
            $SortFieldName = "UserName", $Offset = 0, $Count = 9999999)
    {
        # retrieve matching user IDs
        $UserNames = $this->FindUserNames(
                $SearchString, $FieldName, $SortFieldName, $Offset, $Count);

        # create user objects
        $Users = array();
        foreach ($UserNames as $UserId => $UserName)
        {
            $Users[$UserId] = new User($this->DB, intval($UserId));
        }

        # return array of user objects to caller
        return $Users;
    }

    # return array of user names/IDs who have values matching search string
    # (array indexes are user IDs, array values are user names)
    function FindUserNames($SearchString, $FieldName = "UserName", 
            $SortFieldName = "UserName", $Offset = 0, $Count = 9999999)
    {
        # construct database query (massage search string to use AND logic)
        $QueryString = "SELECT UserId, UserName FROM APUsers WHERE";
        $Words = preg_split("/[\s]+/", trim($SearchString));
        $NewSearchString = "";
        $InQuotedString = FALSE;
        foreach ($Words as $Word)
        {
            if ($InQuotedString == FALSE) {  $NewSearchString .= "+";  }
            if (preg_match("/^\"/", $Word)) {  $InQuotedString = TRUE;  }
            if (preg_match("/\"$/", $Word)) {  $InQuotedString = FALSE;  }
            $NewSearchString .= $Word." ";
        }
        $QueryString .= " MATCH (".$FieldName.")"
                ." AGAINST ('".addslashes(trim($NewSearchString))."'"
                ." IN BOOLEAN MODE)";
        $QueryString .= " ORDER BY ".$SortFieldName
                ." LIMIT ".$Offset.", ".$Count;

        # retrieve matching user IDs
        $this->DB->Query($QueryString);
        $UserNames = $this->DB->FetchColumn("UserName", "UserId");

        # return names/IDs to caller
        return $UserNames;
    }

    # return array of users who have values matching search string (in specific field if requested)
    # (search string respects POSIX-compatible regular expressions)
    # optimization: $SearchString = ".*." and $FieldName = NULL will return all
    #   users ordered by $SortFieldName
    function GetMatchingUsers($SearchString, $FieldName = NULL, 
                              $SortFieldName = "UserName",
                              $ResultsStartAt = 0, $ReturnNumber = NULL)
    {
        # start with empty array (to prevent array errors)
        $ReturnValue = array();

        # if search string supplied
        if (strlen(trim($SearchString)) > 0)
        {
            # build the sorting clause
            $QueryOrderBy = " ORDER BY $SortFieldName";

            if ($ReturnNumber)
            {
                $QueryLimit = " LIMIT ";
                if ($ResultsStartAt)
                {
                    $QueryLimit .= "$ResultsStartAt, $ReturnNumber";
                }
                else
                    $QueryLimit .= $ReturnNumber;
            }
            else
            {
                $QueryLimit = "";
            }

            $Query = "SELECT * FROM APUsers";

            # the Criteria Query will be used to get the total number
            # of results without the limit clause
            $CriteriaQuery = $Query;


            # if specific field comparison requested
            if ($FieldName != NULL)
            {
                # append queries with criteria
                $Query .= " WHERE ".$FieldName." REGEXP '".addslashes($SearchString)."'";

                $CriteriaQuery = $Query;

                # tack on ordering and limiting
                $Query .= $QueryOrderBy.$QueryLimit;

            }
            # optimize for returning all users
            elseif (strcasecmp($SearchString, ".*.") == 0)
            {
                $Query .= $QueryOrderBy.$QueryLimit;

                # set field name to username - this would be the first field
                # returned by a field to field search using the above RegExp
                $FieldName = "UserName";
            }
            else
            {
                # search all fields - can't limit here, but we can order by
                $Query .= $QueryOrderBy;
            }

            # execute query
            $this->DB->Query($Query);

            # process query return
            while ($Record = $this->DB->FetchRow())
            {

                # if specific field or all users requested
                if ($FieldName != NULL)
                {
                    # add user to return array
                    $ReturnValue[$Record["UserId"]] = $Record;

                    # add matching search field to return array
                    $ReturnValue[$Record["UserId"]]["APMatchingField"] = $FieldName;
                }
                else
                {
                    # for each user data field
                    foreach ($Record as $FName => $FValue)
                    {
                        # if search string appears in data field
                        #if (ereg($SearchString, $Record[$FName]))
						if (preg_match($SearchString, $Record[$FName]))
                        {
                            # add user to return array
                            $ReturnValue[$Record["UserId"]] = $Record;

                            # add matching search field to return array
                            $ReturnValue[$Record["UserId"]]["APMatchingField"] = $FName;

                            # move on to next user
                            continue;
                        }
                    }

                    # cut return array down to requested size
                    if (isset($ReturnNumber))
                    {
                        $ReturnValue = array_slice($ReturnValue, $ResultsStartAt, $ReturnNumber, true);
                    }
                }
            }

            if (!isset($this->MatchingUserCount))
            {
                $this->DB->Query($CriteriaQuery);
                $this->MatchingUserCount = $this->DB->NumRowsSelected();
            }

        }

        # return array of matching users to caller
        return $ReturnValue;
    }

    # check whether user name currently exists
    function UserNameExists($UserName)
    {
        # normalize user name
        $UserName = User::NormalizeUserName($UserName);

        # check whether user name is already in use
        $NameCount = $this->DB->Query(
                "SELECT COUNT(*) AS NameCount FROM APUsers"
                    ." WHERE UserName = '".addslashes($UserName)."'",
                "NameCount");

        # report to caller whether name exists
        return ($NameCount > 0);
    }

    # check whether e-mail address currently has account associated with it
    function EMailAddressIsInUse($Address)
    {
        # normalize address
        $UserName = User::NormalizeEMailAddress($Address);

        # check whether address is already in use
        $AddressCount = $this->DB->Query(
                "SELECT COUNT(*) AS AddressCount FROM APUsers"
                    ." WHERE EMail = '".addslashes($Address)."'",
                "AddressCount");

        # report to caller whether address is in use
        return ($AddressCount > 0);
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $DB;
    var $Session;
    var $SortFieldName;
    var $MatchingUserCount;

    # callback function for sorting users
    function CompareUsersForSort($UserA, $UserB)
    {
        return strcasecmp($UserA[$this->SortFieldName], $UserB[$this->SortFieldName]);
    }
};


?>
