<?PHP
#
#   FILE:  SearchParameterSetEditingUI.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2016 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Class to create a user interface for editing SearchParameterSets.
*/
class SearchParameterSetEditingUI
{
    /**
    * Create a UI for specifing edits to SearchParameterSets.
    * @param string $FormFieldName HTML 'name' to use for <input>
    *   elements created by the UI.  If this UI is incorporated into a
    *   form containing other input elements, they must have names
    *   that differ from this one.
    * @param SearchParameterSet $SearchParams SearchParameterSet to display
    *  (OPTIONAL, uses an empty set if unspecified)
    */
    public function __construct($FormFieldName, $SearchParams = NULL)
    {
        $this->EditFormName = $FormFieldName;

        if ($SearchParams !== NULL)
        {
            $this->SearchParams = $SearchParams;
        }
        else
        {
            $this->SearchParams = new SearchParameterSet();
        }

        # get the list of fields that are allowed in searches for all schemas
        $this->MFields = array();
        $this->AllSchemas = MetadataSchema::GetAllSchemas();
        foreach ($this->AllSchemas as $SCId => $Schema)
        {
            foreach ($Schema->GetFields(NULL, MetadataSchema::MDFORDER_ALPHABETICAL)
                     as $FId => $Field)
            {
                if ($Field->IncludeInAdvancedSearch() ||
                    $Field->IncludeInKeywordSearch() )
                {
                    $this->MFields[]= $Field;
                }
            }
        }

        $this->Factories = array();
    }

    /**
    * Display editing form elements enclosed in a <table>.  Note that
    * it still must be wrapped in a <form> that has a submit button.
    * @param string $TableId HTML identifier to use (OPTIONAL, default
    *   NULL).
    * @param string $TableStyle CSS class to attach for this table
    *   (OPTIONAL, default NULL).
    */
    public function DisplayAsTable($TableId = NULL, $TableStyle = NULL)
    {
        print('<table id="'.defaulthtmlentities($TableId).'" '
              .'class="'.defaulthtmlentities($TableStyle).'" '
              .'style="width: 100%">');
        $this->DisplayAsRows();
        print('</table>');
    }

    /**
    * Display the table rows for the editing form, without the
    * surrounding <table> tags.
    */
    public function DisplayAsRows()
    {

        $Fields = $this->FlattenSearchParams(
            $this->SearchParams);

        # make sure the necessary javascript is required
        $GLOBALS["AF"]->RequireUIFile("jquery-ui.js");
        $GLOBALS["AF"]->RequireUIFile("CW-QuickSearch.js");
        $GLOBALS["AF"]->RequireUIFile("SearchParameterSetEditingUI.js");

        # note that all of the fields we create for these rows will be named
        # $this->EditFormName.'[]' , combining them all into an array of results per
        #   http://php.net/manual/en/faq.html.php#faq.html.arrays

        # css classes required by our javascript are logic_row
        # field-row, and field-value-edit

        $Depth = 0;

        foreach ($Fields as $FieldRow)
        {
            if (is_string($FieldRow) && $FieldRow == "(")
            {
                $Depth++;
                print('<tr><td colspan=2 style="padding-left: 2em;">'
                      .'<input type="hidden" name="'.$this->EditFormName.'[]" '
                      .'value="X-BEGIN-SUBGROUP-X"/>'
                      .'<table class="cw-speui-subgroup">');
            }
            elseif (is_string($FieldRow) && $FieldRow == ")")
            {
                $Depth--;
                $this->PrintTemplateRow();
                print('<input type="hidden" name="'.$this->EditFormName.'[]" '
                    .'value="X-END-SUBGROUP-X"/></table></td></tr>');
            }
            elseif (is_array($FieldRow) && isset($FieldRow["Logic"]))
            {
                print('<tr class="logic_row '.$this->EditFormName.'">'
                      .'<td colspan="3">'
                      .($Depth==0?'Top-Level Logic: ':'Subgroup with '));

                $ListName = $this->EditFormName."[]";
                $Options = array("AND"=>"AND", "OR"=>"OR");
                $SelectedValue = $FieldRow["Logic"];

                $OptList = new HtmlOptionList($ListName, $Options, $SelectedValue);
                $OptList->ClassForList("logic");
                $OptList->PrintHtml();

                print (($Depth>0?' Logic':'').'</td></tr>');
            }
            elseif (is_array($FieldRow) && isset($FieldRow["FieldId"]) )
            {
                $FieldId = $FieldRow["FieldId"];
                $Values = $FieldRow["Values"];
                foreach ($Values as $CurVal)
                {
                    print('<tr class="field-row '.$this->EditFormName.'""
                                ." style="white-space: nowrap;">'
                          ."<td><span class=\"cw-button cw-button-elegant "
                          ."cw-button-constrained cw-speui-delete\">X</span>"
                          ."</td><td>");

                    # for selectable fields, we need to generate all the
                    # html elements that we might need and then depend on
                    # javascript to display only those that are relevant

                    # each field will have four elements

                    # 1. a field selector
                    $this->PrintFieldSelector($FieldId);

                    # 2. a value selector (for option and flag values)
                    $this->PrintValueSelector($FieldId, $CurVal);

                    $SearchText = (StdLib::strpos($CurVal, "=")===0) ?
                                StdLib::substr($CurVal, 1) : $CurVal;

                    # 3. a text entry
                    print('<input type="text" class="field-value-edit" '
                          .'name="'.$this->EditFormName.'[]" '
                          .'placeholder="(search terms)" '
                          .'value="'.defaulthtmlentities($SearchText).'">');

                    # 4. an ajax search box
                    $this->PrintQuicksearch($FieldId, $SearchText);

                    print("</td></tr>");
                }
            }
        }

        # add a template row, used for adding new fields
        $this->PrintTemplateRow();
    }

    /**
    * Extract values from a dynamics field edit/modification form.
    * @return SearchParameterSet extracted from post data.  If POST
    *   contains no data, an empty SearchParameterSet will be returned.
    */
    public function GetValuesFromFormData()
    {
        if (!isset($_POST[$this->EditFormName]))
        {
            $Result = new SearchParameterSet();
        }
        else
        {
            # set up our result
            $GroupStack = array();
            array_push($GroupStack, new SearchParameterSet() );

            # extract the array of data associated with our EditFormName
            $FormData = $_POST[$this->EditFormName];

            # extract and set the search logic, which is always the first
            # element in the HTML that we generate
            $Logic = array_shift($FormData);
            end($GroupStack)->Logic($Logic);

            while (count($FormData))
            {
                # first element of each row is a field id
                $FieldId = array_shift($FormData);

                if ($FieldId == "X-BEGIN-SUBGROUP-X")
                {
                    # add a new subgroup to our stack of subgroups
                    array_push($GroupStack, new SearchParameterSet());
                    # extract and set the search logic
                    $Logic = array_shift($FormData);
                    end($GroupStack)->Logic($Logic);
                }
                elseif ($FieldId == "X-END-SUBGROUP-X")
                {
                    $Subgroup = array_pop($GroupStack);
                    end($GroupStack)->AddSet($Subgroup);
                }
                else
                {
                    # for selectable fields, we'll have all possible
                    # elements and will need to grab the correct ones for
                    # the currently selected field
                    $SelectVal = array_shift($FormData);
                    $TextVal   = array_shift($FormData);
                    $SearchVal = array_shift($FormData);

                    if ($FieldId == "X-KEYWORD-X")
                    {
                        $Val = $TextVal;
                        $Field = NULL;

                        if (strlen($TextVal)==0)
                        {
                            continue;
                        }
                    }
                    else
                    {
                        $Field = new MetadataField($FieldId);

                        # make sure we have factories for field types that need them
                        switch ($Field->Type())
                        {
                            case MetadataSchema::MDFTYPE_TREE:
                            case MetadataSchema::MDFTYPE_CONTROLLEDNAME:
                            case MetadataSchema::MDFTYPE_OPTION:
                                if (!isset($this->Factories[$FieldId]))
                                {
                                    $this->Factories[$FieldId] = $Field->GetFactory();
                                }
                                break;

                            default:
                                break;
                        }

                        # verify that we actually have a value for our selected field
                        switch ($Field->Type())
                        {
                            case MetadataSchema::MDFTYPE_PARAGRAPH:
                            case MetadataSchema::MDFTYPE_URL:
                            case MetadataSchema::MDFTYPE_TEXT:
                            case MetadataSchema::MDFTYPE_NUMBER:
                            case MetadataSchema::MDFTYPE_DATE:
                            case MetadataSchema::MDFTYPE_TIMESTAMP:
                            case MetadataSchema::MDFTYPE_USER:
                                # if we have no value for this field, skip displaying it
                                if (strlen($TextVal)==0)
                                {
                                    continue 2;

                                }
                                break;

                            case MetadataSchema::MDFTYPE_TREE:
                            case MetadataSchema::MDFTYPE_CONTROLLEDNAME:
                                # if we have no value for this field, skip displaying it
                                if (strlen($SearchVal)==0)
                                {
                                    continue 2;
                                }
                                break;

                            # no need to check the types where there's
                            # a SelectVal, as that cannot be left empty
                            default:
                                break;
                        }

                        # extract the value for our field
                        switch ($Field->Type())
                        {
                            case MetadataSchema::MDFTYPE_PARAGRAPH:
                            case MetadataSchema::MDFTYPE_URL:
                            case MetadataSchema::MDFTYPE_TEXT:
                            case MetadataSchema::MDFTYPE_NUMBER:
                            case MetadataSchema::MDFTYPE_DATE:
                            case MetadataSchema::MDFTYPE_TIMESTAMP:
                                $Val = $TextVal;
                                break;

                            case MetadataSchema::MDFTYPE_USER:
                                $Val = "=".$TextVal;
                                break;

                            case MetadataSchema::MDFTYPE_TREE:
                                $Item = $this->Factories[$FieldId]->GetItem(
                                    $SearchVal);

                                # for tree fields, use the same 'is X
                                # or a child of X' construction that we
                                # use when generating search facets
                                $Val = new SearchParameterSet();
                                $Val->Logic("OR");
                                $Val->AddParameter(array(
                                    "=".$Item->Name(),
                                    "^".$Item->Name()." -- "), $Field);
                                break;

                            case MetadataSchema::MDFTYPE_CONTROLLEDNAME:
                                $Item = $this->Factories[$FieldId]->GetItem(
                                    $SearchVal);
                                $Val = "=".$Item->Name();
                                break;

                            case MetadataSchema::MDFTYPE_OPTION:
                                list($InputId, $InputVal) = explode("-", $SelectVal, 2);
                                $Item = $this->Factories[$FieldId]->GetItem(
                                    $InputVal);
                                $Val = "=".$Item->Name();
                                break;

                            case MetadataSchema::MDFTYPE_FLAG:
                                list($InputId, $InputVal) = explode("-", $SelectVal, 2);
                                $Val = "=".$InputVal;
                                break;

                            default:
                                throw new Exception("Unsupported field type");
                        }
                    }

                    # add our value to the search parameters
                    if ($Val instanceof SearchParameterSet)
                    {
                        end($GroupStack)->AddSet($Val);
                    }
                    else
                    {
                        end($GroupStack)->AddParameter($Val, $Field);
                    }
                }
            }

            $Result = array_pop($GroupStack);
        }

        $this->SearchParams = $Result;
        return $Result;
    }

    /**
    * Get/Set search parameters.
    * @param SearchParameterSet|null $SearchParams New setting (OPTIONAL)
    * @return Current SearchParameterSet
    */
    public function SearchParameters($SearchParams = NULL)
    {
        if ($SearchParams !== NULL)
        {
            $this->SearchParams = clone $SearchParams;
        }

        return clone $this->SearchParams;
    }

    /**
    * Get/set the max number of characters a label of a field option list
    *        will be displayed.
    * @param int $NewValue Max length of a field option list's label. Use
    *       zero for no limit (OPTIONAL, default to no limit).
    *       If NULL is passed in, this function will not set a new max
    *       length of a field option list.
    * @return Current maximum length of a field option list's label.
    *       Zero means no limit.
    */
    public function MaxFieldLabelLength($NewValue = NULL)
    {
        if (!is_null($NewValue))
        {
            $this->MaxFieldLabelLength = $NewValue;
        }
        return $this->MaxFieldLabelLength;
    }

    /**
    * Get/set the max number of characters a label of a value option list
    *       will be displayed.
    * @param int $NewValue Max length of a field option list's label. Use
    *       zero for no limit (OPTIONAL, default to no limit).
    *       If NULL is passed in, this function will not set a new max
    *       length of a value option list.
    * @return Current maximum length of a value option list's label.
    *       Zero means no limit.
    */
    public function MaxValueLabelLength($NewValue = NULL)
    {
        if (!is_null($NewValue))
        {
            $this->MaxValueLabelLength = $NewValue;
        }
        return $this->MaxValueLabelLength;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $EditFormName;
    private $SearchParams;
    private $MFields;
    private $AllSchemas;
    private $Factories;
    private $MaxFieldLabelLength = 0;
    private $MaxValueLabelLength = 0;

    /**
    * Convert a SearchParameterSet into a flat array that can be
    * easily iterated over when outputting HTML form elements.
    * @param SearchParameterSet $SearchParams Paramters to convert
    * @return array where each element is one of
    *   array("Logic" => LogicSetting)
    *   array("FieldId" => FieldValues)
    *   array("X-KEYWORD-X" => KeywordValues)
    *   "("   -- denoting the beginning of a subgroup
    *   ")"   -- denoting the end of a subgroup
    */
    private function FlattenSearchParams($SearchParams)
    {
        $Result = array();

        $Result[]= array(
            "Logic" => $SearchParams->Logic() );

        $SearchStrings = $SearchParams->GetSearchStrings();
        foreach ($SearchStrings as $FieldId => $Values)
        {
            $Result[]= array(
                "FieldId" => $FieldId,
                "Values" => $Values);
        }

        $KeywordStrings = $SearchParams->GetKeywordSearchStrings();
        if (count($KeywordStrings))
        {
            $Result[]= array(
                "FieldId" => "X-KEYWORD-X",
                "Values" => $KeywordStrings);
        }

        $Subgroups = $SearchParams->GetSubgroups();
        if (count($Subgroups))
        {
            foreach ($Subgroups as $Subgroup)
            {
                $Result[]= "(";
                $SubgroupItems = $this->FlattenSearchParams($Subgroup);
                foreach ($SubgroupItems as $Item)
                {
                    $Result[] = $Item;
                }
                $Result[]= ")";
            }
        }

        return $Result;
    }

    /**
    * Print HTML elements for the field selector.
    * @param string $FieldId Currently selected field.
    */
    private function PrintFieldSelector($FieldId)
    {
        $ListName = $this->EditFormName."[]";
        $SelectedValue = array();

        # "Keyword" option is always here
        $Options["X-KEYWORD-X"] = "Keyword";
        $OptionClass["X-KEYWORD-X"] = "field-type-keyword";
        if ($FieldId == "X-KEYWORD-X")
        {
            $SelectedValue[] = "X-KEYWORD-X";
        }

        # prepare options for print
        foreach ($this->MFields as $MField)
        {
            $TypeName = defaulthtmlentities(
                str_replace(' ', '', strtolower($MField->TypeAsName())));

            if (!$MField->Optional())
            {
                $TypeName .= " required";
            }

            $FieldName = $MField->Name();
            if ($MField->SchemaId() != MetadataSchema::SCHEMAID_DEFAULT)
            {
                $FieldName = $this->AllSchemas[$MField->SchemaId()]->Name()
                           .": ".$FieldName;
            }

            $Options[$MField->Id()] = defaulthtmlentities($FieldName);
            $OptionClass[$MField->Id()] = "field-type-".$TypeName;

            if ($FieldId == $MField->Id())
            {
                $SelectedValue[] = $MField->Id();
            }
        }

        # instantiate option list and print
        $OptList = new HtmlOptionList($ListName, $Options, $SelectedValue);
        $OptList->ClassForList("field-subject");
        $OptList->ClassForOptions($OptionClass);
        $OptList->MaxLabelLength($this->MaxFieldLabelLength);
        $OptList->PrintHtml();
    }


    /**
    * Print HTML elements for the value selector for Option and Flag fields.
    * @param int $FieldId Currently selected FieldId.
    * @param string $CurVal Currently selected value.
    */
    private function PrintValueSelector($FieldId, $CurVal)
    {
        # parameters of the option list
        $ListName = $this->EditFormName."[]";
        $Options = array();
        $OptionClass = array();
        $SelectedValue = array();

        # prepare options for print
        foreach ($this->MFields as $MField)
        {
            if ($MField->Type() == MetadataSchema::MDFTYPE_FLAG ||
                $MField->Type() == MetadataSchema::MDFTYPE_OPTION)
            {
                foreach ($MField->GetPossibleValues() as $Id => $Val)
                {
                    $IsSelected = FALSE;
                    $Key = $MField->Id()."-".$Id;

                    if ($MField->Id() == $FieldId)
                    {
                        $TgtVal = ($MField->Type() == MetadataSchema::MDFTYPE_FLAG) ?
                            "=".$Id : "=".$Val ;
                        $IsSelected = ($CurVal == $TgtVal) ? TRUE : FALSE ;
                    }

                    $Options[$Key] = defaulthtmlentities($Val);
                    $OptionClass[$Key] = "field-id-".$MField->Id();

                    if ($IsSelected)
                    {
                        $SelectedValue[] = $Key;
                    }
                }
            }
        }

        # instantiate an option list and print
        $OptList = new HtmlOptionList($ListName, $Options, $SelectedValue);
        $OptList->ClassForList("field-value-select");
        $OptList->ClassForOptions($OptionClass);
        $OptList->MaxLabelLength($this->MaxValueLabelLength);
        $OptList->PrintHtml();
    }

    /**
    * Output quicksearch field for ControlledName and Tree fields.
    * @param int $FieldId Currently selected FieldId.
    * @param string $CurVal Currently selected field value.
    */
    private function PrintQuicksearch($FieldId, $CurVal)
    {
        if ($FieldId !== NULL && $FieldId != "X-KEYWORD-X")
        {
            if (!isset($this->Factories[$FieldId]))
            {
                $Field = new MetadataField($FieldId);
                $Factory = $Field->GetFactory();

                $this->Factories[$FieldId] =
                        ($Factory !== NULL) ? $Factory : FALSE;
            }

            $ItemId = ($this->Factories[$FieldId] !== FALSE) ?
                    $this->Factories[$FieldId]->GetItemIdByName($CurVal) : "" ;
        }
        else
        {
            $ItemId = "";
        }

        # field-value-qs css class required by our javascript.
        # various cw-quicksearch classes required by the quicksearch javascript
        # and ui-front class required by jquery-ui, used by qs js
        print('<div class="field-value-qs cw-quicksearch '
              . 'cw-quicksearch-template">'
              .'<input class="cw-quicksearch-display '
              .'cw-resourceeditor-metadatafield" '
              .'placeholder="(enter some text to begin searching)" '
              .'value="'.defaulthtmlentities($CurVal).'" />'
              .'<input name="'.$this->EditFormName.'[]" '
              .'class="cw-quicksearch-value" type="hidden" '
              .'value="'.defaulthtmlentities($ItemId).'" />'
              .'<div style="display: none;" '
              .'class="cw-quicksearch-menu">'
              .'<div class="cw-quicksearch-message ui-front"></div>'
              .'</div></div>');
    }

    /**
    * Output template row for JS to copy when new fields are added.
    */
    private function PrintTemplateRow()
    {
        # field-row, template-row, field-value-edit, cw-speui-add, and
        # cw-speui-add-subgroup css classes required by our javascript
        print(
            "<tr class=\"field-row template-row ".$this->EditFormName."\""
                    ." style=\"white-space: nowrap;\">"
            ."<td>"
            ."<span class=\"cw-button cw-button-elegant cw-button-constrained "
            ."cw-speui-delete\">X</span>"
            ."</td><td>");
        $this->PrintFieldSelector(NULL);
        $this->PrintValueSelector(NULL, "");
        print("<input type=\"text\" class=\"field-value-edit\" "
              ."name=\"".$this->EditFormName."[]\" placeholder=\"(search terms)\" "
              ."value=\"\">");
        $this->PrintQuicksearch(NULL, "");
        print("</td></tr>");
        print("<tr><td colspan=2>"
              ."<span class=\"cw-button cw-button-elegant cw-button-constrained "
              ."cw-speui-add\">Add Field</span>"
              ."<span class=\"cw-button cw-button-elegant cw-button-constrained "
              ."cw-speui-add-subgroup\">Add Subgroup</span>"
              ."</td></tr>");
    }
}
