<?PHP
#
#   FILE:  File.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2010-2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Class representing a stored (usually uploaded) file.
*/
class File extends Item
{

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # status codes (set by constructor and returned by File::Status())
    const FILESTAT_OK =             0;
    const FILESTAT_COPYERROR =      1;
    const FILESTAT_PARAMERROR =     2;
    const FILESTAT_ZEROLENGTH =     3;
    const FILESTAT_DOESNOTEXIST =   4;
    const FILESTAT_UNREADABLE =     5;

    /**
    * Create a new File object using an existing file.
    * @param string $SourceFile Name of existing file, with absolute or
    *       relative leading path, if needed.
    * @param string $DesiredFileName Desired name for file (if not the same
    *       as the existing name).  (OPTIONAL).
    * @return mixed New File object or error code if creation failed.
    */
    public static function Create($SourceFile, $DesiredFileName = NULL)
    {
        # check that file exists
        if (!file_exists($SourceFile))
        {
            return self::FILESTAT_DOESNOTEXIST;
        }

        # check that file is readable
        if (!is_readable($SourceFile))
        {
            return self::FILESTAT_UNREADABLE;
        }

        # check that file is not zero length
        $FileSize = filesize($SourceFile);
        if (!$FileSize)
        {
            return self::FILESTAT_ZEROLENGTH;
        }

        # generate secret string (used to protect from unauthorized download)
        srand((double)microtime() * 1000000);
        $SecretString = sprintf("%04X", rand(1, 30000));

        # get next file ID by adding file to database
        $DB = new Database();
        $DB->Query("INSERT INTO Files (SecretString) VALUES ('".$SecretString."')");
        $FileId = $DB->LastInsertId();

        # build name for stored file
        $BaseFileName = ($DesiredFileName === NULL)
                ? basename($SourceFile) : basename($DesiredFileName);
        $StoredFile = sprintf(self::GetStorageDirectory()."/%06d-%s-%s",
                $FileId, $SecretString, $BaseFileName);

        # attempt to copy file to storage
        $Result = copy($SourceFile, $StoredFile);

        # if copy attempt failed
        if ($Result === FALSE)
        {
            # remove file from database
            $DB->Query("DELETE FROM Files WHERE FileId = ".$FileId);

            # report error to caller
            return self::FILESTAT_COPYERROR;
        }

        # attempt to get file type
        $FileType = self::DetermineFileType($SourceFile);

        # save file info in database
        $DB->Query("UPDATE Files SET"
                ." FileName = '".addslashes($BaseFileName)."',"
                ." FileType = '".addslashes($FileType)."',"
                ." FileLength = '".addslashes($FileSize)."'"
                ." WHERE FileId = ".$FileId);

        # instantiate new object and return it to caller
        return new File($FileId);
    }

    /**
    * Create copy of File object.  The copy will have a new ID, but will
    * otherwise be identical.
    * @return object Copy of object.
    */
    public function CreateCopy()
    {
        $Copy = self::Create($this->GetNameOfStoredFile(), $this->Name());
        if (!$Copy instanceof self)
        {
            throw new Exception("Copy failed with error ".$Copy);
        }
        $Copy->ResourceId($this->ResourceId());
        $Copy->FieldId($this->FieldId());
        return $Copy;
    }

    /**
    * Gets the length of the file.
    * @return The length of the file.
    */
    public function GetLength()
    {
        return $this->ValueCache["FileLength"];
    }

    /**
    * Gets the file's type.
    * @return The file's type.
    */
    public function GetType()
    {
        return $this->ValueCache["FileType"];
    }

    /**
    * Gets or sets the comment on the file.
    * @param string $NewValue The new comment on the file.  (OPTIONAL)
    * @return The comment on the file.
    */
    public function Comment($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("FileComment", $NewValue);
    }

    /**
    * Gets or sets the field ID of the File.
    * @param int $NewValue The new field ID of the File.  (OPTIONAL)
    * @return The field ID of the File.
    */
    public function FieldId($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("FieldId", $NewValue);
    }

    /**
    * Gets or sets the resource ID of the File.
    * @param int $NewValue The new resource ID of the File.  (OPTIONAL)
    * @return The resource ID of the File.
    */
    public function ResourceId($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ResourceId", $NewValue);
    }

    /**
    * Gets the MIME type of the file.
    * @return The MIME type of the file.
    */
    public function GetMimeType()
    {
        return strlen($this->GetType())
                ? $this->GetType() : "application/octet-stream";
    }

    /**
    * Returns the relative download link to download the file. If .htaccess
    * files are supported, the redirect that includes the file name is used.
    * @return The relative link to download the file.
    */
    public function GetLink()
    {
        # if CleanURLs are enabled, use the redirect that includes
        # the file name so that browsers don't use index.php as the name
        # for the downloaded file
        if ($GLOBALS["G_PluginManager"]->PluginEnabled("CleanURLs"))
        {
            return "downloads/".$this->Id."/".rawurlencode($this->Name());
        }

        # otherwise use the download portal
        else
        {
            return "index.php?P=DownloadFile&Id=".$this->Id;
        }
    }

    /**
    * Deletes the file and removes its entry from the database. Other methods
    * are invalid after calling this.
    */
    public function Destroy()
    {
        # delete file
        $FileName = $this->GetNameOfStoredFile();
        if (file_exists($FileName))
        {
            unlink($FileName);
        }

        # call parent method
        parent::Destroy();
    }

    /**
    * Deprecated method to delete file and remove entry from database.
    * @see Destroy()
    * @deprecated
    */
    public function Delete()
    {
        $this->Destroy();
    }

    /**
    * Returns the relative link to the stored file.
    * @return The relative link to the stored file
    */
    public function GetNameOfStoredFile()
    {
        # for each possible storage location
        foreach (self::$StorageLocations as $Dir)
        {
            # build file name for that location
            $FileName = sprintf($Dir."/%06d-%s-%s",
                    $this->Id, $this->ValueCache["SecretString"], $this->Name());

            # if file can be found in that location
            if (file_exists($FileName))
            {
                # return file name to caller
                return $FileName;
            }
        }

        # build file name for default (most preferred) location
        $FileName = sprintf(self::GetStorageDirectory()."/%06d-%s-%s",
                $this->Id, $this->ValueCache["SecretString"], $this->Name());

        # return file name to caller
        return $FileName;
    }

    /**
    * Get file storage directory.
    * @return string Relative directory path (with no trailing slash).
    */
    public static function GetStorageDirectory()
    {
        # for each possible storage location
        foreach (self::$StorageLocations as $Dir)
        {
            # if location exists and is writeable
            if (is_dir($Dir) && is_writeable($Dir))
            {
                # return location to caller
                return $Dir;
            }
        }

        # return default (most preferred) location to caller
        return self::$StorageLocations[0];
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    /** File storage directories, in decreasing order of preference. */
    static private $StorageLocations = array(
            "local/data/files",
            "FileStorage",
            );

    /**
    * Get MIME type for specified file, if possible.
    * @param string $FileName Name of file, with absolute or relative
    *       leading path, if needed.
    * @return string MIME type, or empty string if unable to determine type.
    */
    protected static function DetermineFileType($FileName)
    {
        $FileType = "";
        if (function_exists("mime_content_type"))
        {
            $FileType = mime_content_type($FileName);
        }
        # Although mime_content_type is baked into PHP5 and PHP7, it is still
        # part of extensions in some versions of PHP 5.x.
        elseif (function_exists("finfo_open"))
        {
            $FInfoHandle = finfo_open(FILEINFO_MIME);
            if ($FInfoHandle)
            {
                $Result = finfo_file($FInfoHandle, $FileName);
                finfo_close($FInfoHandle);
                if ($Result)
                {
                    $FileType = $FInfoMime;
                }
            }
        }

        # handle Office XML formats
        # These are recognized by PHP as zip files (because they are), but
        # IE (and maybe other things?) need a special-snowflake MIME type to
        # handle them properly.
        # For a list of the required types, see
        # https://technet.microsoft.com/en-us/library/ee309278(office.12).aspx
        if ($FileType == "application/zip; charset=binary")
        {
            $MsftPrefix = "application/vnd.openxmlformats-officedocument";

            $FileExt = strtolower(pathinfo($FileName, PATHINFO_EXTENSION));

            switch ($FileExt)
            {
                case "docx":
                    $FileType = $MstfPrefix.".wordprocessingml.document";
                    break;

                case "xlsx":
                    $FileType = $MsftPrefix.".spreadsheetml.sheet";
                    break;

                case "pptx":
                    $FileType = $MsftPrefix.".presentationml.slideshow";
                    break;

                default:
                    # do nothing
            }
        }

        return $FileType;
    }
}
