<?PHP
#
#   FILE:  Forum.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2002-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* A converastion forum which includes topics and messages.
* @nosubgrouping
*/
class Forum extends Item
{

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /** @name Setup/Initialization/Destruction */
    /*@{*/

    /**
    * Create a forum.
    * @param string $Name Name of the forum.
    * @param int $ModeratorId UserId of the forum moderator.
    * @param str $Description Description of the forum
    *       (OPTIONAL, defaults to none).
    * @return New forum object.
    */
    public static function Create($Name, $ModeratorId, $Description = NULL)
    {
        $DB = new Database();
        $ModeratorId = intval($ModeratorId);

        # name is required
        $Name = trim($Name);
        if (!strlen($Name))
        {
            throw new InvalidArgumentException("Specified ForumName is empty.");
        }

        # check if the ID provided is a valid user ID and has privilege
        $Factory = new UserFactory();
        if ($Factory->UserExists($ModeratorId))
        {
            $Moderator = new CWUser($ModeratorId);
            if (!$Moderator->HasPriv(PRIV_FORUMADMIN))
            {
                throw new InvalidArgumentException("Specified moderator ID ("
                        .$ModeratorId.") is not a forum administrator.");
            }
        }
        else
        {
            throw new InvalidArgumentException("Specified moderator ID ("
                    .$ModeratorId.") is not found.");
        }

        # set description if there is any
        if (!is_null($Description))
        {
            $Query = "INSERT INTO Forums (ForumName, ModeratorId, ForumDescription)".
                " VALUES ('".addslashes($Name)."', '".addslashes($ModeratorId)."', '"
                .addslashes($Description)."')";
        }
        else
        {
            $Query = "INSERT INTO Forums (ForumName, ModeratorId) VALUES ('"
                    .addslashes($Name)."', '".addslashes($ModeratorId)."')";
        }

        # create a new record
        $DB->Query($Query);

        # get its forum id
        $ForumId = $DB->LastInsertId();

        return (new Forum($ForumId));
    }

    /**
    * Remove this forum, deleting all assocated topics and messages.
    */
    public function Delete()
    {
        $this->DB->Query("SELECT * FROM Topics WHERE ForumId = ".
                         $this->Id()." ORDER BY DateCreated DESC");

        # get list of topics for this forum
        while ($Entry = $this->DB->FetchRow())
        {
            $Topic = new Topic($Entry["TopicId"]);
            $Topic->Delete();
        }
        # delete record from database
        $this->Destroy();
    }
    /*@}*/

    /** @name Accessors */
    /*@{*/

    /**
    * Get the forum's ID.
    * @return Returns the forum ID
    * @deprecated Use Id() instead.
    */
    public function ForumId()
    {
        return $this->Id();
    }

    /**
    * Get the date of the most recent post to the forum.
    * @return Returns the date of the most recent post or "None" for empty
    *      forums
    */
    public function LastMessageDate()
    {
        $Message = $this->GetLastMessage();
        if (isset($Message))
        {
            return GetPrettyTimestamp($Message->DatePosted())." by ";
        }
        else
        {
            return "None";
        }
    }

    /**
    * Get the CWIS username of the user with the most recent post.
    * @return Returns the CWIS username of the user with the most recent post.
    */
    public function LastMessagePoster()
    {
        $Message = $this->GetLastMessage();
        if (isset($Message))
        {
            return $Message->PosterName();
        }
    }

    /**
    * Get the e-mail address of the user with the most recent post.
    * @return Returns the e-mail address of the user with the most recent post.
    */
    public function LastMessagePosterEmail()
    {
        $Message = $this->GetLastMessage();
        if (isset($Message))
        {
            return $Message->PosterEmail();
        }
    }

    /**
    * Get the CWIS username of the forum's moderator.
    * @return Returns the CWIS username of the forum's moderator
    *       or NULL if there is no such user with the UserId.
    */
    public function ModeratorName()
    {
        $UserId = (int)$this->ModeratorId();
        $UserFactory = new CWUserFactory();

        if ($UserFactory->UserExists($UserId))
        {
            return (new CWUser($UserId))->Get("UserName");
        }
        else
        {
            return NULL;
        }
    }

    /**
    * Get the e-mail address of the forum's moderator.
    * @return Returns the e-mail address of the forum's modreator
    *       or NULL if there is no such user with the UserId.
    */
    public function ModeratorEmail()
    {
        $UserId = (int)$this->ModeratorId();
        $UserFactory = new CWUserFactory();

        if ($UserFactory->UserExists($UserId))
        {
            return (new CWUser($UserId))->Get("EMail");
        }
        else
        {
            return NULL;
        }
    }

    /**
    * Get the list of the topics in this forum.
    * @return Returns an array of topic IDs for this forum's conversations.
    */
    public function GetTopicList()
    {
        $Topics = array();

        $this->DB->Query("SELECT * FROM Topics WHERE ForumId = '".
                    addslashes($this->Id())."' ORDER BY DateCreated DESC");

        # get list of topics for this forum
        while ($Entry = $this->DB->FetchRow())
        {
            $Topics[$Entry["TopicId"]] = new Topic($Entry["TopicId"]);
        }
        return $Topics;
    }

    /**
    * Get the last message posted in the forum.
    * @return Returns a Message object of the last posted message or NULL if
    *      there isn't one.
    */
    public function GetLastMessage()
    {
        $Message = NULL;

        $this->DB->Query("
            SELECT M.* FROM Messages M
            LEFT JOIN Topics T
            ON M.ParentId = T.TopicId
            WHERE M.ParentType = ".Message::PARENTTYPE_TOPIC."
            AND T.ForumId = '".addslashes($this->Id())."'
            ORDER BY DatePosted DESC
            LIMIT 1");

        if ($this->DB->NumRowsSelected())
        {
            $Row = $this->DB->FetchRow();
            $Message = new Message($Row["MessageId"]);
        }

        return $Message;
    }

    /**
    * Get or modify the forum's name.
    * @param string $NewValue New forum name. This parameter is optional.
    * @return Returns the current forum name.
    */
    public function ForumName($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ForumName", $NewValue);
    }

    /**
    * Get or modify the forum description.
    * @param string $NewValue New forum description. This parameter is optional.
    * @return Returns the current forum description.
    */
    public function ForumDescription($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ForumDescription", $NewValue);
    }

    /**
    * Get or set the forum's topic count.
    * @param int $NewValue New forum topic count. This parameter is optional.
    * @return Returns the forum's current topic count.
    */
    public function TopicCount($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("TopicCount", $NewValue);
    }

    /**
    * Get or set the forum's message count.
    * @param int $NewValue New forum message count. This parameter is optional.
    * @return Returns the current message count.
    */
    public function MessageCount($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("MessageCount", $NewValue);
    }

    /**
    * Get or set the forum's moderator.
    * @param int $NewValue New forum moderator's user ID. This parameter is
    *      optional.
    * @return Returns the CWIS user ID of the forum's moderator.
    */
    public function ModeratorId($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ModeratorId", $NewValue);
    }

    /*@}*/

    /**
    * Add topic to forum.
    * @param CWUser $Author User adding new topic.
    * @param string $TopicName Name of new topic.
    * @param string $Subject Subject for new topic.
    * @param string $Body Text body for new topic.
    * @return int ID of new topic.
    */
    public function AddTopic($Author, $TopicName, $Subject, $Body)
    {
        $Topic = Topic::Create();
        $Topic->TopicName($TopicName);
        $Topic->DateCreated(date("YmdHis"));
        $Topic->ForumId($this->Id());
        $Topic->CreatorId($Author->Id());

        $this->TopicCount($this->TopicCount() + 1);

        $this->PostMessage($Topic->TopicId(), $Author, $Subject, $Body);

        return $Topic->TopicId();
    }

    /**
    * Post new message to topic.
    * @param int $TopicId ID of topic.
    * @param CWUser $Author User adding new message.
    * @param string $Subject Subject for new message.
    * @param string $Body Text body for new message.
    */
    public function PostMessage($TopicId, $Author, $Subject, $Body )
    {
        $Topic = new Topic($TopicId);

        # create a new message
        $Message = Message::Create();
        $Message->ParentId($TopicId);
        $Message->ParentType(Message::PARENTTYPE_TOPIC);
        $Message->DatePosted(date("YmdHis"));
        $Message->PosterId($Author->Id());
        $Message->Subject($Subject);
        $Message->Body($Body);

        # update counts for topic and forum
        $this->MessageCount($this->MessageCount() + 1 );

        $Topic->MessageCount($Topic->MessageCount() + 1);

    }

    /**
    * Delete a message from a forum, updating the message counts for
    * the associated forum and topic.
    * @param int $MessageId ID of the message to delete
    */
    public static function DeleteMessage($MessageId)
    {
        $Message = new Message($MessageId);

        $TopicId = $Message->ParentId();
        $Topic = new Topic($TopicId);

        $ForumId = $Topic->ForumId();
        $Forum = new Forum($ForumId);

        # update count for topic and forum and delete message
        $Topic->MessageCount($Topic->MessageCount() - 1 );
        $Forum->MessageCount($Forum->MessageCount() - 1 );

        $Message->Delete();
    }
}
