<?PHP
#
#   FILE:  EditRule.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

# ----- LOCAL FUNCTIONS ------------------------------------------------------

/**
* Check that a PrivilegeSet includes user-related components.
* @param string $FieldName Name of form field for which value is being validated.
* @param array $FieldValues Form field value(s).
* @return string NULL if value is okay or error message if not.
*/
function CheckThatPrivilegesHaveUserComponents($FieldName, $FieldValues)
{
    $PrivSet = $FieldValues;
    return (count($PrivSet->GetPossibleNecessaryPrivileges())
            || count($PrivSet->FieldsWithUserComparisons())) ? NULL
            : "The conditions for <i>Email Recipients</i> must include at"
                    ." least one user privilege or user field comparison,"
                    ." to limit the potential recipient list.";
}


/**
* Construct the list of fields that we want included in our privsets.
* @return array of fields to include.
*/
function GetFieldsForPrivset()
{
    $PrivFields = array();
    foreach (MetadataSchema::GetAllSchemas() as $SchemaId => $Schema)
    {
        # for the User schema
        if ($SchemaId == MetadataSchema::SCHEMAID_USER)
        {
            # supported types, in the order they should be displayed
            $SupportedFieldTypesInOrder = array(
                MetadataSchema::MDFTYPE_USER,
                MetadataSchema::MDFTYPE_FLAG,
                MetadataSchema::MDFTYPE_OPTION,
                MetadataSchema::MDFTYPE_TIMESTAMP,
                MetadataSchema::MDFTYPE_NUMBER);

            # fields we want to skip
            # UserId is included because UserId = UserId of Recipient
            # will always be true
            $UserFieldsToExclude = array("UserId");

            # iterate over each type
            foreach ($SupportedFieldTypesInOrder as $Type)
            {
                # get fields of that type
                foreach ($Schema->GetFields($Type) as $FieldId => $Field)
                {
                    # if this field should be excluded, skip it
                    if (in_array($Field->Name(), $UserFieldsToExclude) )
                    {
                        continue;
                    }
                    # otherwise add it to the list
                    $PrivFields[$FieldId] = $Field;
                }
            }
        }
        else
        {
            # for all other schemas, add all the User fields
            foreach ($Schema->GetFields(MetadataSchema::MDFTYPE_USER)
                     as $FieldId => $Field)
            {
                $PrivFields[$FieldId] = $Field;
            }
        }
    }

    return $PrivFields;
}

# ----- MAIN -----------------------------------------------------------------

# check permissions
CheckAuthorization(PRIV_SYSADMIN, PRIV_COLLECTIONADMIN);

# retrieve rule ID
$H_RuleId = GetFormValue("ID");
if ($H_RuleId === NULL)
{
    throw new Exception("No rule ID specified.");
}
$H_IsNewRule = ($H_RuleId == "NEW") ? TRUE : FALSE;

# if editing existing rule
if (!$H_IsNewRule)
{
    # load rule
    $H_Rule = new Rules_Rule($H_RuleId);
}

# set up editing form
$FormFields = array(
        "Rule Heading" => array(
                "Type" => FormUI::FTYPE_HEADING,
                "Label" => "Rule",
                ),
        "Enabled" => array(
                "Type" => FormUI::FTYPE_FLAG,
                "Label" => "Enabled",
                "Default" => TRUE,
                ),
        "Name" => array(
                "Type" => FormUI::FTYPE_TEXT,
                "Label" => "Name",
                "Placeholder" => "(rule name)",
                "Required" => TRUE,
                ),
        "Frequency" => array(
                "Type" => FormUI::FTYPE_OPTION,
                "Label" => "Check Frequency",
                "Help" => "How often to check for items that match the rule.",
                "Options" => array(
                        60 => "Hourly",
                        240 => "Every 4 Hours",
                        480 => "Every 8 Hours",
                        1440 => "Daily",
                        10080 => "Weekly",
                        0 => "Continuously",
                        ),
                "Default" => 60,
                ),
        # ------------------------------------------------
        "Condition Heading" => array(
                "Type" => FormUI::FTYPE_HEADING,
                "Label" => "If",
                ),
        "SearchParams" => array(
                "Type" => FormUI::FTYPE_SEARCHPARAMS,
                "Label" => "Search Parameters",
                "Help" => "Search parameters that need to be met for items"
                        ." to match the rule.",
                "Required" => TRUE,
                "MaxFieldLabelLength" => 45,
                "MaxValueLabelLength" => 25
                ),
        # ------------------------------------------------
        "Action Heading" => array(
                "Type" => FormUI::FTYPE_HEADING,
                "Label" => "Then",
                ),
        "Action" => array(
                "Type" => FormUI::FTYPE_OPTION,
                "Label" => "Action",
                "Help" => "What to do when new items that match the rule"
                        ." are found.",
                "Options" => array(
                        ),
                "Required" => TRUE,
                ),
        );

# if creating a new rule
if ($H_IsNewRule)
{
    # start with empty values
    $FormValues = array(
            "SearchParams" => new SearchParameterSet(),
            );
}
else
{
    # load existing values
    $FormValues = array(
            "Name" => $H_Rule->Name(),
            "Enabled" => $H_Rule->Enabled(),
            "Frequency" => $H_Rule->CheckFrequency(),
            "SearchParams" => $H_Rule->SearchParameters(),
            "Action" => $H_Rule->Action(),
            );
}

# if support for "Send Email" action is available
if ($GLOBALS["G_PluginManager"]->PluginEnabled("Mailer"))
{
    # add additional action option
    $FormFields["Action"]["Options"][Rules_Rule::ACTION_SENDEMAIL] = "Send Email";

    # add additional settings for action
    $MailerPlugin = $GLOBALS["G_PluginManager"]->GetPlugin("Mailer");
    $FormFields["SendEmail_Template"] = array(
            "Type" => FormUI::FTYPE_OPTION,
            "Label" => "Email Template",
            "Help" => "The template to use when sending emails.",
            "Options" => $MailerPlugin->GetTemplateList(),
            );
    $FormFields["SendEmail_Privileges"] = array(
            "Type" => FormUI::FTYPE_PRIVILEGES,
            "Label" => "Email Recipients",
            "Help" => "Emails will be sent to users"
                    ." for whom specified conditions are satisfied.",
            "Required" => TRUE,
            "ValidateFunction" => "CheckThatPrivilegesHaveUserComponents",
            "Schemas" => array(),
            "MetadataFields" => GetFieldsForPrivset(),
            );
    $FormFields["SendEmail_ConfirmBeforeSending"] = array(
            "Type" => FormUI::FTYPE_FLAG,
            "Label" => "Email Requires Confirmation",
            "Help" => "If enabled, emails will be queued for confirmation "
                    ." rather than sent immediately.",
    );

    # if adding new rule
    if ($H_IsNewRule)
    {
        # set blank privilege set for recipients for form
        $FormValues["SendEmail_Privileges"] = new PrivilegeSet();
    }
    else
    {
        # retrieve and set existing settings for form
        $ActionParams = $H_Rule->ActionParameters();
        $FormValues["SendEmail_Template"] = $ActionParams["Template"];
        $FormValues["SendEmail_Privileges"] = new PrivilegeSet(
                $ActionParams["Privileges"]);
        $FormValues["SendEmail_ConfirmBeforeSending"] =
                $ActionParams["ConfirmBeforeSending"];
    }

    # set default action if one not already set
    if (!isset($FormFields["Action"]["Default"]))
    {
        $FormFields["Action"]["Default"] = Rules_Rule::ACTION_SENDEMAIL;
    }
}

# instantiate form UI
$H_FormUI = new FormUI($FormFields, $FormValues);

$ButtonPushed = GetFormValue("Submit");
switch ($ButtonPushed)
{
    case "Add":
    case "Save":
        # check values and bail out if any are invalid
        if ($H_FormUI->ValidateFieldInput())
        {
            return;
        }

        # retrieve values from form
        $NewSettings = $H_FormUI->GetNewValuesFromForm();
        $SearchParams = $NewSettings["SearchParams"];
        $Action = $NewSettings["Action"];

        # retrieve action-specific attributes for rule
        switch ($Action)
        {
            case Rules_Rule::ACTION_SENDEMAIL:
                $ActionParams = array(
                        "Template" => $NewSettings["SendEmail_Template"],
                        "Privileges" => $NewSettings["SendEmail_Privileges"]->Data(),
                        "ConfirmBeforeSending" =>
                            $NewSettings["SendEmail_ConfirmBeforeSending"],
                );
                break;
        }

        # if adding new rule
        if ($H_IsNewRule)
        {
            # create new rule
            $H_Rule = Rules_Rule::Create($SearchParams, $Action, $ActionParams);
        }
        else
        {
            # load existing rule
            $H_Rule = new Rules_Rule($H_RuleId);

            # save updated condition and action for rule
            $H_Rule->SearchParameters($SearchParams);
            $H_Rule->Action($Action);
            $H_Rule->ActionParameters($ActionParams);
        }

        # save common new attributes for rule
        $H_Rule->Enabled($NewSettings["Enabled"]);
        $H_Rule->Name($NewSettings["Name"]);
        $H_Rule->CheckFrequency($NewSettings["Frequency"]);

        # perform rule check to update list of IDs that match rule if needed
        if (!$H_IsNewRule)
        {
            $MatchingIds = $H_Rule->Check();
            $H_Rule->LastMatchingIds($MatchingIds);
        }

        # return to rule list
        $GLOBALS["AF"]->SetJumpToPage("P_Rules_ListRules");
        break;

    case "Cancel":
        # return to rule list
        $GLOBALS["AF"]->SetJumpToPage("P_Rules_ListRules");
        break;
}
