<?PHP

#
#   FILE:  SPT--FormTool.php
#
#   METHODS PROVIDED:
#       FormTool()
#           - constructor
#       SomeMethod($SomeParameter, $AnotherParameter)
#           - short description of method
#
#   AUTHOR:  Edward Almasy
#
#   Part of the Scout Portal Toolkit
#   Copyright 2006 Internet Scout Project
#   http://scout.wisc.edu
#

require_once("include/SPT--Common.php");


class FormTool {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function FormTool($FormFields, $AdditionalErrorMessages = NULL)
    {
        # save field info with field name as index
        foreach ($FormFields as $Field)
        {
            $this->Fields[$Field->Name()] = $Field;
        }

        # save additional error messages (if any)
        $this->AdditionalErrorMessages = $AdditionalErrorMessages;

        # set default error color
        $this->ErrorColor = "red";

        # save any additional fields indicated to be marked for error
        if (isset($_GET["FTAddErrFields"]) && count($_GET["FTAddErrFields"]))
        {
            $this->AdditionalErrorFields = explode("!", $_GET["FTAddErrFields"]);
        }
        else
        {
            $this->AdditionalErrorFields = array();
        }
    }

    # get/set field values
    function ValueForField($FieldName, $NewValue = NULL)
    {
        return $this->Fields[$FieldName]->Value($NewValue);
    }

    # checks whether all required form variables are set in $_POST
    function AllRequiredVarsAreSet()
    {
        # assume that all required variables will be found
        $AllFound = TRUE;

        # for each required form variable
        foreach ($this->Fields as $FieldName => $Field)
        {
            # if variable is not set in $_POST
            if (!isset($_POST[$FieldName]) || !strlen($_POST[$FieldName]))
            {
                # set flag indicating we found a missing var and exit loop
                $AllFound = FALSE;
                break;
            }
        }

        # report back to caller whether we found on required vars
        return $AllFound;
    }

    # return URL parameter string with form values that are set (from $_POST)
    # (returns empty string if none of the form values are set)
    # (URL parameter string does not include leading separator (? or &))
    function GetValsAsUrlParams($IncludePasswords = FALSE)
    {
        # assume no values will be found
        $UrlParams = "";
    
        # for each form field
        foreach ($this->Fields as $FieldName => $Field)
        {
            # if form value is set and contains something and is not excluded
            if (isset($_POST[$FieldName]) 
                    && strlen(trim($_POST[$FieldName]))
                    && (!$Field->IsPassword() || $IncludePasswords))
            {
                # add value to URL param string
                $UrlParams = strlen($UrlParams)
                        ? $UrlParams."&".$FieldName."=".urlencode($_POST[$FieldName])
                        : $FieldName."=".urlencode($_POST[$FieldName]);
            }
        }
    
        # return parameter string to caller
        return $UrlParams;
    }

    # set field values from URL parameters where available
    function SetFieldValuesFromUrlParams()
    {
        # for each field
        foreach ($this->Fields as $FieldName => $Field)
        {
            # if value is available for field in incoming GET parameters
            if (isset($_GET[$FieldName]))
            {
                # set field value
                $Field->Value($_GET[$FieldName]);
            }
        }
    }

    # check form values for each field and report whether errors found
    function IncomingFieldValuesHaveErrors()
    {
        return (count($_GET) || count($_POST)) ?
            (strlen($this->GetErrorCodesAsUrlParams()) ? TRUE : FALSE) : FALSE;
    }

    # return URL parameter string with codes for any form value errors
    # (returns empty string if no errors found)
    # (URL parameter string does not include leading separator (? or &))
    function GetErrorCodesAsUrlParams()
    {
        # start with empty error code string
        $ErrorCodeString = "";

        # for each field value
        foreach ($this->Fields as $FieldName => $Field)
        {
            # if validation function says that value is invalid
            $ErrorCode = $this->CheckFieldValue($FieldName);
            if ($ErrorCode)
            {
                # add error code for value to error code string
                $ErrorCodeString .= (strlen($ErrorCodeString) ? "!" : "")
                        .$FieldName."-".$ErrorCode;
            }
        }

        # if values were added to error code string
        if (strlen($ErrorCodeString))
        {
            # prepend name of GET variable to contain error codes
            $ErrorCodeString = "FTFieldErrs=".$ErrorCodeString;
        }

        # if additional error codes were supplied
        if (isset($this->AdditionalErrorCodes) && count($this->AdditionalErrorCodes))
        {
            # for each additional error code
            foreach ($this->AdditionalErrorCodes as $Code)
            {
                # append code to string
                $AddCodeString = isset($AddCodeString) ? $AddCodeString."!".$Code
                        : $Code;
            }

            # append additional error code string to error code string
            $ErrorCodeString .= (strlen($ErrorCodeString) ? "&" : "")
                    ."FTAddErrCodes=".$AddCodeString;
        }

        # if additional fields were supplied to be marked as erroneous
        if (count($this->AdditionalErrorFields))
        {
            # for each additional error code
            foreach ($this->AdditionalErrorFields as $FieldName)
            {
                # append code to string
                $AddFieldString = isset($AddFieldString) ? $AddFieldString."!".$FieldName
                        : $FieldName;
            }

            # append additional error code string to error code string
            $ErrorCodeString .= (strlen($ErrorCodeString) ? "&" : "")
                    ."FTAddErrFields=".$AddFieldString;
        }

        # return error code string to caller
        return $ErrorCodeString;
    }

    # save additional fields to be marked as having errors
    function SetAdditionalErrorFields($FieldNames)
    {
        # convert to array if needed
        if (!is_array($FieldNames))
        {
            $FieldNames = array($FieldNames);
        }

        # save fields (if not already present)
        foreach ($FieldNames as $FieldName)
        {
            if (!in_array($FieldName, $this->AdditionalErrorFields))
            {
                $this->AdditionalErrorFields[] = $FieldName;
            }
        }
    }

    # save additional error codes
    function SetAdditionalErrorCodes($Codes)
    {
        # convert to array if needed
        if (!is_array($Codes))
        {
            $Codes = array($Codes);
        }

        # save codes (if not already present)
        foreach ($Codes as $Code)
        {
            if (!isset($this->AdditionalErrorCodes)
                    || !in_array($Code, $this->AdditionalErrorCodes))
            {
                $this->AdditionalErrorCodes[] = $Code;
            }
        }
    }

    # convenience function that adds value and err codes to URL
    function GetUrlWithValuesAndErrorCodes($BaseUrl, $IncludePasswords = FALSE)
    {
        $ValParams = $this->GetValsAsUrlParams($IncludePasswords);
        $ErrParams = $this->GetErrorCodesAsUrlParams();
        return $BaseUrl
                .(strlen($ValParams) ? "?".$ValParams : "")
                .(strlen($ErrParams) ? 
                        (strlen($ValParams) ? "&" : "?").$ErrParams : "");
    }

    # get list of error messages based on codes from URL ($_GET)
    function GetErrorMessages($EliminateDuplicateMessages = TRUE)
    {
        # start with empty list
        $ErrorList = array();

        # if it looks like there are field-specific error messages to be had
        if (isset($_GET["FTFieldErrs"]))
        {
            # split error data into list of fields
            $FieldList = explode("!", $_GET["FTFieldErrs"]);

            # for each field found
            foreach ($FieldList as $FieldListEntry)
            {
                # split field entry into name and code
                list($FieldName, $ErrorCode) = explode("-", $FieldListEntry);

                # if we know about this field
                if (isset($this->Fields[$FieldName]))
                {
                    # translate error code into message and add to list
                    $Field = $this->Fields[$FieldName];
                    $Replacements = array(
                            "%N" => "<i>".$Field->Name()."</i>",
                            "%V" => "<i>".$Field->Value()."</i>",
                            "%L" => "<i>".preg_replace("/:$/", "", $Field->Label())."</i>",
                            "%C" => "<i>".$ErrorCode."</i>",
                            );
                    $Message = $Field->GetInvalidValueMessage($ErrorCode);
                    $ErrorList[$FieldName] = str_replace(
                            array_keys($Replacements), $Replacements, $Message);
                }
            }
        }

        # if it looks like there are additional general error messages to be had
        if (isset($_GET["FTAddErrCodes"]) && count($this->AdditionalErrorMessages))
        {
            # split error data into list of codes
            $CodeList = explode("!", $_GET["FTAddErrCodes"]);

            # for each code found
            foreach ($CodeList as $Code)
            {
                # if there is a message corresponding to this code
                if (isset($this->AdditionalErrorMessages[$Code]))
                {
                    # add message to list
                    $ErrorList[$Code] = $this->AdditionalErrorMessages[$Code];
                }
            }
        }

        # remove duplicate messages (if requested by caller)
        if ($EliminateDuplicateMessages)
        {
            $NewErrorList = array();
            foreach ($ErrorList as $Code => $Message)
            {
                if (!in_array($Message, $NewErrorList))
                {
                    $NewErrorList[$Code] = $Message;
                }
            }
            $ErrorList = $NewErrorList;
        }

        # return list of error messages to caller
        return $ErrorList;
    }

    # print tags for specified field
    function PrintField($FieldName)
    {
        $this->Fields[$FieldName]->PrintField(
                ($this->ErrorCodesAvailable() && $this->CheckFieldValue($FieldName))
                || in_array($FieldName, $this->AdditionalErrorFields));
    }
    function PrintLabelForField($FieldName)
    {
        $this->Fields[$FieldName]->PrintLabel(
                ($this->ErrorCodesAvailable() && $this->CheckFieldValue($FieldName))
                || in_array($FieldName, $this->AdditionalErrorFields));
    }
    function PrintInputForField($FieldName)
    {
        $this->Fields[$FieldName]->PrintInput(
                ($this->ErrorCodesAvailable() && $this->CheckFieldValue($FieldName))
                || in_array($FieldName, $this->AdditionalErrorFields));
    }

    # report whether error codes are available
    function ErrorCodesAvailable()
    {
        return isset($_GET["FTFieldErrs"]) || isset($_GET["FTAddErrCodes"]);
    }

    # return array of US state names with two-letter abbreviations as index
    # (may be called statically)
    function GetArrayOfUsStates()
    {
        return array(
                "" => "--",
                "AL" => "Alabama",
                "AK" => "Alaska",
                "AZ" => "Arizona",
                "AR" => "Arkansas",
                "CA" => "California",
                "CO" => "Colorado",
                "CT" => "Connecticut",
                "DE" => "Delaware",
                "DC" => "District of Columbia",
                "FL" => "Florida",
                "GA" => "Georgia",
                "HI" => "Hawaii",
                "ID" => "Idaho",
                "IL" => "Illinois",
                "IN" => "Indiana",
                "IA" => "Iowa",
                "KS" => "Kansas",
                "KY" => "Kentucky",
                "LA" => "Louisiana",
                "ME" => "Maine",
                "MD" => "Maryland",
                "MA" => "Massachusetts",
                "MI" => "Michigan",
                "MN" => "Minnesota",
                "MS" => "Mississippi",
                "MO" => "Missouri",
                "MT" => "Montana",
                "NE" => "Nebraska",
                "NV" => "Nevada",
                "NH" => "New Hampshire",
                "NJ" => "New Jersey",
                "NM" => "New Mexico",
                "NY" => "New York",
                "NC" => "North Carolina",
                "ND" => "North Dakota",
                "OH" => "Ohio",
                "OK" => "Oklahoma",
                "OR" => "Oregon",
                "PA" => "Pennsylvania",
                "RI" => "Rhode Island",
                "SC" => "South Carolina",
                "SD" => "South Dakota",
                "TN" => "Tennessee",
                "TX" => "Texas",
                "UT" => "Utah",
                "VT" => "Vermont",
                "VA" => "Virginia",
                "WA" => "Washington",
                "WV" => "West Virginia",
                "WI" => "Wisconsin",
                "WY" => "Wyoming",
                );
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $Fields;
    var $ErrorColor;
    var $AdditionalErrorCodes;
    var $AdditionalErrorFields;
    var $AdditionalErrorMessages;

    # check form (POST) value for specified field and return error code
    function CheckFieldValue($FieldName)
    {
        $Value = isset($_POST[$FieldName]) ? $_POST[$FieldName] 
                : (isset($_GET[$FieldName]) ? $_GET[$FieldName] : NULL);
        $ErrorCode = $this->Fields[$FieldName]->IsInvalidValue($Value);
        return $ErrorCode;
    }
}


class TextFormField extends FormField {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function TextFormField(
            $Name, $IsRequired, $Label, $Length, $MaxLength, 
            $ValidFunc = NULL, $ValidMsgs = NULL)
    {
        $this->MyLength = $Length;
        $this->MyMaxLength = $MaxLength;

        $this->FormField($Name, $IsRequired, $Label, $ValidFunc, $ValidMsgs);
    }

    # get/set field attributes
    function Length($NewVal = NULL) {  return $this->GetOrSet("MyLength", $NewVal);  }
    function MaxLength($NewVal = NULL) {  return $this->GetOrSet("MyMaxLength", $NewVal);  }

    # print input tags
    function PrintInput($DisplayErrorIndicator = FALSE)
    {
        print("<input type=\"".
                    # (hack to support PasswordFormField object as well)
                    (method_exists($this, "PasswordFormField") ? "password" : "text")
                    ."\""
                ." name=\"".$this->MyName."\""
                ." value=\"".htmlspecialchars($this->MyValue)."\""
                ." size=\"".$this->MyLength."\""
                ." maxlength=\"".$this->MyMaxLength."\""
                .($DisplayErrorIndicator ? " style=\"background-color: #FFEEEE;\"" : "")
                ." />");
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $MyLength;
    var $MyMaxLength;
}

class PasswordFormField extends TextFormField {
    function PasswordFormField(
            $Name, $IsRequired, $Label, $Length, $MaxLength, 
            $ValidFunc = NULL, $ValidMsgs = NULL)
    {
        $this->TextFormField(
                $Name, $IsRequired, $Label, $Length, $MaxLength, $ValidFunc, $ValidMsgs);
    }
}

class OptionFormField extends FormField {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function OptionFormField(
            $Name, $IsRequired, $Label, $Length, $Options, 
            $ValidFunc = NULL, $ValidMsgs = NULL)
    {
        $this->MyLength = $Length;
        $this->MyOptions = $Options;

        $this->FormField($Name, $IsRequired, $Label, $ValidFunc, $ValidMsgs);
    }

    # get/set field attributes
    function Length($NewVal = NULL) {  return $this->GetOrSet("MyLength", $NewVal);  }
    function Options($NewVal = NULL) {  return $this->GetOrSet("MyOptions", $NewVal);  }

    # print input tags
    function PrintInput($DisplayErrorIndicator = FALSE)
    {
        print("<select name=\"".$this->MyName."\" size=\"".$this->MyLength."\">\n");
        foreach ($this->MyOptions as $OptionValue => $OptionLabel)
        {
            print("    <option value=\"".htmlspecialchars($OptionValue)."\""
                    .(($OptionValue == $this->Value()) ? " selected" : "")
                    .">".htmlspecialchars($OptionLabel)."\n");
        }
        print("</select>\n");
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $MyLength;
    var $MyOptions;
}

class CheckboxFormField extends FormField {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function CheckboxFormField($Name, $Label)
    {
        $this->FormField($Name, FALSE, $Label, NULL, NULL);
    }

    # print label and input tags
    function PrintField($DisplayErrorIndicator = FALSE)
    {
        $this->PrintInput($DisplayErrorIndicator);
        $this->PrintLabel($DisplayErrorIndicator);
    }

    # print input tags
    function PrintInput($DisplayErrorIndicator = FALSE)
    {
        print("<input type=\"checkbox\""
                ." name=\"".$this->MyName."\""
                ." id=\"".$this->MyName."\""
                .($this->MyValue ? " checked" : "")
                .">\n");
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

}

class FormField {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function FormField($Name, $IsRequired, $Label, $ValidFunc, $ValidMsgs)
    {
        # save field info
        $this->MyName = $Name;
        $this->MyIsRequired = $IsRequired;
        $this->MyLabel = $Label;
        $this->MyValidFunc = $ValidFunc;
        $this->MyValidMsgs = $ValidMsgs;

        # attempt to set value if available
        if (isset($_POST[$this->MyName]))
        {
            $this->MyValue = $_POST[$this->MyName];
        }
        elseif (isset($_GET[$this->MyName]))
        {
            $this->MyValue = $_GET[$this->MyName];
        }
    }

    # get/set field attributes
    function Name($NewVal = NULL) {  return $this->GetOrSet("MyName", $NewVal);  }
    function IsRequired($NewVal = NULL) {  return $this->GetOrSet("MyIsRequired", $NewVal);  }
    function IsPassword() {  return method_exists($this, "PasswordFormField");  }
    function Label($NewVal = NULL) {  return $this->GetOrSet("MyLabel", $NewVal);  }
    function Value($NewVal = NULL) {  return $this->GetOrSet("MyValue", $NewVal);  }

    # print label and input tags
    function PrintField($DisplayErrorIndicator = FALSE)
    {
        $this->PrintLabel($DisplayErrorIndicator);
        $this->PrintInput($DisplayErrorIndicator);
    }

    # print label tags
    function PrintLabel($DisplayErrorIndicator = FALSE)
    {
        # print label
        print(($DisplayErrorIndicator ? "<span style=\"color: red;\"" : "")
            ."<label for=\"".$this->MyName."\">".$this->MyLabel."</label>"
            .($DisplayErrorIndicator ? "</span>" : "")
            ."\n");
    }

    # check possible field value for validity
    # (returns non-zero error code if invalid)
    # (this method is intended to be overloaded by some child classes)
    function IsInvalidValue($Value)
    {
        # assume value is valid
        $ErrorCode = 0;

        # if custom validation function supplied
        if ($this->MyValidFunc)
        {
            # call custom function and return code
            $ValidFunc = $this->MyValidFunc;
            $ErrorCode = $ValidFunc($this->MyName, $Value);
        }
        else
        {
            # if value is required and none is set
            if ($this->MyIsRequired && !strlen($Value) 
                    && !method_exists($this, "PasswordFormField"))
            {
                # return code indicating missing value
                $ErrorCode = 1;
            }
        }

        # return error code (if any) to caller
        return $ErrorCode;
    }
    # map field validity error codes to text error messages
    # (this assumes that the error code came from FormField::IsInvalidValue())
    function GetInvalidValueMessage($ErrorCode)
    {
        $Messages = array(
                0 => "This value is valid.",
                1 => "%L is a required value.",
                );
        if (isset($this->MyValidMsgs[$ErrorCode]))
        {
            $Message = $this->MyValidMsgs[$ErrorCode];
        }
        else
        {
            $Message = isset($Messages[$ErrorCode]) 
                    ? $Messages[$ErrorCode] : 
                            "INTERNAL ERROR - Invalid Error Code (Field = %N, Code = %C)";
        }
        return $Message;
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $MyName;
    var $MyIsRequired;
    var $MyLabel;
    var $MyValue;
    var $MyValidFunc;
    var $MyValidMsgs;

    # convenience function to handle getting and setting of values
    function GetOrSet($ValueName, $NewValue)
    {
        if ($NewValue !== NULL)
        {
            $this->{$ValueName} = $NewValue;
        }
        return $this->{$ValueName};
    }
}


?>
